#!/bin/bash

# Creates the file spielplan.csv file for import into Google calendar from export of "Staffelspielplan" webpage from fussball.de
# Licence: The usual GNU GPL V3.0 - http://www.gnu.org/copyleft/gpl.html
# Date: July 19, 2012
# Author: Heino Falcke, h.falcke@astro.ru.nl
#
# Usage: 
# Speichere den Staffelspielplan als webarchive in einer Datei und dann tippe vond er Kommandozeile
#
# ./fussball2csv.sh [optionale Parameter] htmlfile teamname 
#
# oder alternativ kann die Website von fussball.de direkt eingegeben werden 
#
# ./fussball2csv.sh [optionale Parameter]  webaddress teamname
#
# oder man wählt die schon vorher benutzte Liga aus der Liste, die bei Aufruf des Programms
# gezeigt wird (und in weblinks.txt gespeichert sind und editiert werden können).
#
# ./fussball2csv.sh [optionale Parameter] number teamname
#
# Positionsgebundene Parameter:
# 
# htmlfile: Name der Datei mit dem gespeicherten webarchive von fussball.de mit dem Spielplan (z.b. ~/Desktop/Staffelspielplan*.webarchiv)
# webaddress: Browser-Link zum Staffelspielplan auf fussball.de 
# number: Zeilennummer in weblinks.txt mit der gewünschten Staffel
#
# teamname: (Teil des) Teamnamen um das Team, dessen Spiele man haben will herauszufiltern (Für die Experten: hier steht eigentlich eine "regular expression/regexp")
#
# Optionale Parameter (müssen als erster Parameter kommen):
#
# -h : Liste Hilfe für die Benutzung des Programms
# -l : Liste nur die existierenden weblinks
# -d directory : specify the output directory where file should be saved
# -o filename  : specify the filename to be used for output
# -s style     : output Stil - entweder CALENDAR (für Google calendar, Default), EXCEL, oder TEXT

 
listonly=0
outputfile=""
outputdir="."
style="CALENDAR"

while getopts ":lho:d:s:" opt; do
  case $opt in
    l)
      listonly=1
      ;;
    o)
      outputfile=$OPTARG
      ;;
    d)
      outputdir=$OPTARG
      ;;
    s)
      style=$OPTARG
      ;;
    h)
      head -n 37 $0
      exit 0
      ;;
    \?)
      echo "Invalid option: -$OPTARG" >&2
      exit 1
      ;;
    :)
      echo "Option -$OPTARG requires an argument." >&2
      exit 1
      ;;
  esac
done

shift $((OPTIND-1))

source=$1
teamname=$2
n=0

outputfile_ext=".csv"
if [ $style = "TEXT" ]; then outputfile_ext=".txt"; fi

#echo Source: $source, Teamname: $teamname, Style: $style, Outputdir: $outputdir

tmpdir=/tmp/fussball2csv
history_weblinks=$tmpdir/weblinks.txt

if [ ! -d $tmpdir ]; then mkdir -p $tmpdir; fi
if [ ! -d ${history_weblinks%/*} ]; then mkdir -p ${history_weblinks%/*}; fi
if [ ! -f $history_weblinks ]; then touch $history_weblinks; fi

echo " "
echo "Schon benutzte Staffelspielpläne - diese können auch über die Zahl gewählt werden:"
echo "=================================================================================="
awk '{N=N+1; sub("http://ergebnisdienst.fussball.de/staffelspielplan/",""); sub("/[^/]*$",""); gsub("/"," - "); print N": " $0}' $history_weblinks 
echo "=================================================================================="

if [ $listonly -eq 1 ]; then exit 0; fi

if [ -f $source ]; then # ist eine Datei
file=$source
else # ist eine Webseite
if [ -z ${source%%[0-9]*} ]; then # ist eine Zahl, also Index der auf die Webseite in der History verweist
n=$source
source=`head -n $source $history_weblinks | tail -1`
else # ist echt ein (neuer) weblink, speichere ihn in der History
if [ -z `grep "$source" $history_weblinks` ]; then echo $source >> $history_weblinks; fi # check if weblink is already in history file
fi

file=$tmpdir/staffelplan.html
./wget -nv -O $file $source

fi 

if [ -z $outputfile ]; then 
ofile1=${file##*/}
ofile=$outputdir/${ofile1%.*}-$n-$style$outputfile_ext
else
ofile=$outputfile
fi

#expand ~ etc.
eval ofile=$ofile

if [ -f $ofile ]; then rm -f $ofile; fi 

echo " "
echo "Konvertiere die Datei" "$file" "ins CSV-Format:"
echo "--------------------------------------------------"
contents=`awk '{gsub("[<]td ","\n<td "); print $0}' "$file" | awk -f fussball2csv.awk -v csvfile="$ofile" -v team=$teamname -v style=$style` 
echo $contents
echo "--------------------------------------------------"

if [ -z $outputfile ]; then 
  ofile2=$outputdir/${contents##*: }-$style$outputfile_ext
  if [ -f $ofile2 ]; then rm -f $ofile2; fi 
  mv $ofile $ofile2
  ofile=$ofile2
fi

if [ $style == EXCEL ]; then echo "Fertig - importieren sie nun die Datei \"$ofile\" in Excel oder Numbers (drag and drop)"; fi
if [ $style == TEXT ]; then echo "Fertig - öffnen sie nun die Datei \"$ofile\" in einem Editor (Word oder textedit)"; fi
if [ $style == CALENDAR ]; then echo "Fertig - importieren sie nun die Datei \"$ofile\" in Google calendar (via Einstellungen/Kalender/Kalender importieren)."; fi

#converting to mac encoding - should not be needed on propper unix systems
iconv -f utf8 -t mac $ofile > $tmpdir/tmp$outputfile_ext
mv -f $tmpdir/tmp$outputfile_ext $ofile

open -R $ofile

#file -I taxofit.csv 
#iconv -f utf-8 -t mac taxofit.csv > x.csv
#
#./fussball2csv.sh 1 JFS -s EXCEL
#
# tar -cvf fussball2csv.tar fussball2csv.awk fussball2csv.sh

