/*-----------------------------------------------------------------*/
/*
 * gpscom
 *
 * Communicates with Trimble GPS devices connected via FTDI USB-serial
 * adapter.  Kernel driver must have already detected device.  We assume it
 * shows up as /dev/ttyUSB0, although this can be changed below.
 *
 * Communication follows the TSIP protocol (see the Trimble Resolution T
 * manual), and the tsip.h header file.  
 *
 * J. Kelley
 * j.kelley@astro.ru.nl
 * August 2009
 *
 */
/*-----------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>

#include "tsip.h"

#define BAUDRATE B9600
#define DEVICE "/dev/ttyUSB0"
#define _POSIX_SOURCE 1 /* POSIX compliant source */
#define FALSE 0
#define TRUE 1

#define MAX_MSG  15000

#define PI 3.1415926535898

volatile int mode, new_mode; 
enum {STARTUP, RUN, GET_PARAMS, WAIT_PARAMS, COMMAND, FLASH, STOP};
enum {CMD_RECEIVE = 0x1, CMD_UTC = 0x2, CMD_DYNAMICS = 0x4, 
      CMD_FLASH = 0x80};

unsigned char param_buf[100];
int param_buf_len = 0;
int dummy;

/*-----------------------------------------------------------------*/

unsigned int get_uint32(unsigned char *m) {
    unsigned int x;
    x = (m[0] << 24) | (m[1] << 16) | (m[2] << 8) | m[3];
    return x;
}

unsigned short get_uint16(unsigned char *m) {
    unsigned short x;
    x = (m[0] << 8) | m[1];
    return x;
}

short get_sint16(unsigned char *m) {
    short x;
    x = (m[0] << 8) | m[1];
    return x;
}

float get_single(unsigned char *m) {
    unsigned int x;
    float f;
    x = (m[0] << 24) | (m[1] << 16) | (m[2] << 8) | m[3];
    f = *((float *)&x);
    return f;
}

double get_double(unsigned char *m) {
    unsigned int x[2];
    double d;
    x[1] = (m[0] << 24) | (m[1] << 16) | (m[2] << 8) | m[3];
    x[0] = (m[4] << 24) | (m[5] << 16) | (m[6] << 8) | m[7];
    d = *((double *)&x);
    return d;
}

/*-----------------------------------------------------------------*/

void tsip_cmd_getconfig(int dev) {
    printf("COMMAND: get configuration\n");
    unsigned char buf[10];
    buf[0] = TSIP_DLE;
    buf[1] = TSIP_CMD_CFG;
    buf[2] = 0x00;
    buf[3] = TSIP_DLE;
    buf[4] = TSIP_ETX;
    dummy = write(dev, buf, 5);
}

/* 
 * Assemble a configuration buffer string with all fields specified 
 * as "no change" (0xFF for UNIT8 and -1.0 for single) 
 */
void tsip_config_buffer(unsigned char *buf) {
    int i;

    /* This has to be zero */
    buf[0] = 0x00;

    /* 1-4: 4x UINT8 */
    for (i = 1; i < 5; i++)
        buf[i] = 0xff;

    /* 5-20: 4x single */
    for (i = 5; i < 21; i++) {
        if (((i-5) % 4) == 0)
            buf[i] = 0xbf;
        else if (((i-5) % 4) == 1)
            buf[i] = 0x80;
        else
            buf[i] = 0;
    }

    /* 21-39: 19x single */
    for (i = 21; i < 40; i++)
        buf[i] = 0xff;
}

/*
 * Set receiver mode (7 = overdetermined clock)
 */
void tsip_cmd_set_rcv_mode(int dev, int rcv_mode) {
    
    unsigned char buf[50];
    if ((rcv_mode >= 0) && (rcv_mode <= 7) && (rcv_mode != 2)) {
        printf("COMMAND: set receiver mode to %d\n", rcv_mode);
        buf[0] = TSIP_DLE;
        buf[1] = TSIP_CMD_CFG;
        tsip_config_buffer(&buf[2]);
        buf[3] = rcv_mode & 0xff;
        buf[42] = TSIP_DLE;
        buf[43] = TSIP_ETX;
        dummy = write(dev, buf, 44);
    }
    else {
        fprintf(stderr, "ERROR: illegal reciever mode %d, command ignored.", rcv_mode);
    }
}

/*
 * Set receiver mode (4 = stationary)
 */
void tsip_cmd_set_dynamics_mode(int dev, int dyn_mode) {
    
    unsigned char buf[50];
    if ((dyn_mode >= 1) && (dyn_mode <= 4)) {
        printf("COMMAND: set dynamics mode to %d\n", dyn_mode);
        buf[0] = TSIP_DLE;
        buf[1] = TSIP_CMD_CFG;
        tsip_config_buffer(&buf[2]);
        buf[5] = dyn_mode & 0xff;
        buf[42] = TSIP_DLE;
        buf[43] = TSIP_ETX;
        dummy = write(dev, buf, 44);
    }
    else {
        fprintf(stderr, "ERROR: illegal dynamics mode %d, command ignored.", dyn_mode);
    }
}

/*
 * Set UTC/GPS mode 
 */
void tsip_cmd_set_utc_mode(int dev, int utc_mode) {

    unsigned char buf[10];

    if ((utc_mode >= 0) && (utc_mode <= 1)) {
        printf("COMMAND: set UTC mode to %d\n", utc_mode);
        buf[0] = TSIP_DLE;
        buf[1] = TSIP_CMD_EXT;
        buf[2] = TSIP_CMD_EXT_UTC;
        if (utc_mode != 0)
            buf[3] = 0x3;
        else
            buf[3] = 0x0;
        buf[4] = TSIP_DLE;
        buf[5] = TSIP_ETX;
        dummy = write(dev, buf, 6);   
    }
    else {
        fprintf(stderr, "ERROR: illegal UTC/GPS mode %d, command ignored.", utc_mode);
    }
}

/*
 * Write to flash and reset 
 */
void tsip_cmd_flash(int dev) {
    printf("COMMAND: save to flash\n");
    unsigned char buf[10];
    buf[0] = TSIP_DLE;
    buf[1] = TSIP_CMD_EXT;
    buf[2] = TSIP_CMD_EXT_SAVE;
    buf[3] = TSIP_DLE;
    buf[4] = TSIP_ETX;
    dummy = write(dev, buf, 5);
}

/*
 * Decode TSIP message and print to stdout.  Return type (or subtype
 * if extended) of message decoded.
 */
unsigned char tsip_decode(unsigned char *m, int len) {

    unsigned char type, subtype;
    unsigned short us;
    int i;

    if (len < 4) {
        fprintf(stderr, "ERROR: message too short!\n");
        return 0;
    }
    
    type = m[1];
    subtype = m[2];

    switch (type) {
    case TSIP_REP_PERR:
        printf("REPORT: parse error\n");
        break;

    case TSIP_REP_SXYZ:
        printf("REPORT: single precision XYZ position\n");
        break;

    case TSIP_REP_VFIX:
        printf("REPORT: velocity fix\n");
        break;
        
    case TSIP_REP_VERS:
        printf("REPORT: software version\n");
        break;
        
    case TSIP_REP_SIG:
        printf("REPORT: signal level\n");
        break;

    case TSIP_REP_SLLA:
        printf("REPORT: single precision LLA position\n");
        break;

    case TSIP_REP_IO:
        printf("REPORT: I/O options\n");
        break;

    case TSIP_REP_VENU:
        printf("REPORT: velocity fix (ENU)\n");
        break;

    case TSIP_REP_CFIX:
        printf("REPORT: last computed fix\n");
        break;

    case TSIP_REP_DATA:
        printf("REPORT: system data / ack\n");
        break;

    case TSIP_REP_RAW:
        printf("REPORT: raw measurement data\n");
        break;
            
    case TSIP_REP_SAT:
        printf("REPORT: satellite tracking status\n");
        break;

    case TSIP_REP_VIEW:
        printf("REPORT: all-in-view satellite selection\n");
        break;

    case TSIP_REP_DXYZ:
        printf("REPORT: double-precision XYZ position\n");
        break;

    case TSIP_REP_DLLA:
        printf("REPORT: double-precision LLA position\n");
        break;
        
    case TSIP_REP_CFG:
        printf("REPORT: primary configuration\n");
        printf(" receiver mode: %d\n", m[3]);
        printf(" dynamics: %d\n", m[5]);
        printf(" elevation: %f\n", get_single(&m[7]));
        printf(" AMU mask: %f\n", get_single(&m[11]));
        printf(" PDOP mask: %f\n", get_single(&m[15]));
        printf(" foliage mode: %d\n", m[24]);
        break;

    case TSIP_REP_PORT:
        printf("REPORT: port configuration\n");
        break;

    case TSIP_REP_EXT:
        switch (subtype) {
        case TSIP_REP_EXT_DATE:
            printf("REPORT: current date\n");
            break;
        case TSIP_REP_EXT_MANU: 
            printf("REPORT: manufacturing parameters\n");
            break;

        case TSIP_REP_EXT_PROD: 
            printf("REPORT: production parameters\n");
            break;

        case TSIP_REP_EXT_PPS:  
            printf("REPORT: set PPS characteristics\n");
            break;

        case TSIP_REP_EXT_UTC:
            printf("REPORT: UTC/GPS timing\n");
            break;

        case TSIP_REP_EXT_TEST:
            printf("REPORT: test modes\n");
            break;

        case TSIP_REP_EXT_POUT:
            printf("REPORT: PPS output options\n");
            break;

        case TSIP_REP_EXT_BCST:
            printf("REPORT: packet broadcast mask\n");
            break;

        case TSIP_REP_EXT_SCMD:
            printf("REPORT: self-survey command\n");
            break;

        case TSIP_REP_EXT_SPRM:
            printf("REPORT: self-survey parameters\n");
            break;
            
        case TSIP_REP_EXT_TIME:
            printf("REPORT: primary timing\n");
            printf(" Week: %u  TOW: %u s\n", get_uint16(&m[7]), get_uint32(&m[3]));
            printf(" UTC offset: %d s\n", get_sint16(&m[9]));
            printf(" UTC/GPS time: %s\n", m[11] & 0x1  ? "UTC" : "GPS");
            printf(" UTC/GPS PPS:  %s\n", m[11] & 0x2  ? "UTC" : "GPS");
            printf(" Time is set:  %s\n", m[11] & 0x4  ? "NO" : "YES");
            printf(" UTC info OK:  %s\n", m[11] & 0x8  ? "NO" : "YES");
            printf(" Time from GPS:%s\n", m[11] & 0x10 ? "NO" : "YES");
            printf(" %d-%d-%d %02d:%02d:%02d\n", m[15], m[16], 
                   get_uint16(&m[17]), m[14], m[13], m[12]);
            break;

        case TSIP_REP_EXT_STIM:
            printf("REPORT: secondary timing\n");
            printf(" Receiver mode %d: ", m[3]);
            switch (m[3]) {
            case 0: 
                printf("automatic");
                break;
            case 1:
                printf("single satellite (time)");
                break;
            case 3:
                printf("horizontal (2D)");
                break;
            case 4:
                printf("full position (3D)");
                break;
            case 5:
                printf("DGPS reference");
                break;
            case 6:
                printf("clock hold");
                break;
            case 7:
                printf("overdetermined clock");
                break;
            default:
                printf("UNKNOWN");
                break;
            }
            printf("\n");
            printf(" self-survey progress: %d%%\n", m[5]);
            us = get_uint16(&m[12]);
            if (us != 0) {
                printf(" minor alarms:\n");
                if (us & 0x2)
                    printf("  antenna open\n");
                if (us & 0x4)
                    printf("  antenna shorted\n");
                if (us & 0x8)
                    printf("  not tracking\n");
                if (us & 0x20)
                    printf("  survey in progress\n");
                if (us & 0x40)
                    printf("  no stored position\n");
                if (us & 0x80)
                    printf("  leap second pending\n");
                if (us & 0x100)
                    printf("  in test mode\n");
                if (us & 0x200)
                    printf("  position is questionable\n");
                if (us & 0x800)
                    printf("  almanac not complete\n");
                if (us & 0x1000)
                    printf("  PPS was generated\n");
            }
            printf(" GPS decoding status: ");
            switch (m[14]) {
            case 0:
                printf("doing fixes\n");
                break;
            case 1:
                printf("don't have GPS time\n");
                break;
            case 3:
                printf("PDOP is too high\n");
                break;
            case 8:
                printf("no usable sats\n");
                break;
            case 9:
                printf("only 1 usable sat\n");
                break;
            case 0xa:
                printf("only 2 usable sats\n");
                break;
            case 0xb:
                printf("only 3 usable sats\n");
                break;
            case 0xc:
                printf("chosen sat is unusable\n");
                break;
            case 0x10:
                printf("TRAIM rejected the fix\n");
                break;
            }
            printf(" local clock bias: %f ns\n", get_single(&m[18]));
            printf(" local clock bias rate: %f ppb\n", get_single(&m[22]));
            printf(" temperature: %.2f C\n", get_single(&m[34]));
            printf(" latitude: %.6f'N\n", get_double(&m[38])*180.0/PI);
            printf(" longitude: %6f'E\n", get_double(&m[46])*180.0/PI);
            printf(" altitude: %g m\n", get_double(&m[54]));
            printf(" PPS quantization error: %f s\n", get_single(&m[62]));
        }
        break;
    default:
        printf("REPORT: unknown type 0x%02x\n", type);
        printf("MSG: len %d\n", len);
        for (i = 0; i < len; i++) {
            printf("%02x ", m[i]);
            if ((i+1)%10 == 0)
                printf("\n");
        }
        printf("\n");        
        break;

    }

    if (type == TSIP_REP_EXT)
        return subtype;
    else
        return type;
}

/*-----------------------------------------------------------------*/
void usage(char *name) {
    fprintf(stderr, "Usage: %s [-p] [device options] [-s <dev>]\n", name);
    fprintf(stderr, "      -p       : read primary configuration and exit\n");
    fprintf(stderr, "      -s <dev> : GPS serial device (default: /dev/ttyUSB0)\n");
    fprintf(stderr, "\n");
    fprintf(stderr, " Device options:\n");
    fprintf(stderr, "      -r <#>   : set receiver mode (0-7; 7=clock mode)\n"); 
    fprintf(stderr, "      -d <#>   : set dynamics mode (1-4; 4=stationary)\n"); 
    fprintf(stderr, "      -u <#>   : set UTC mode (1=on, 0=off/GPS time)\n");
    fprintf(stderr, "      -f       : write settings to flash and reset\n");
}

/*
 * Signal handler for CTRL-C, to quit. 
 */
void quit(int sig) {
    new_mode = STOP;
}

/*-----------------------------------------------------------------*/

int main(int argc, char **argv) {

    struct sigaction sa_old;
    struct sigaction sa_new;
    int i, fd, res, optc;
    struct termios oldtio,newtio;
    unsigned char buf[255];
    unsigned char c, lastc, lastlastc, type;
    unsigned char m[MAX_MSG];
    int mIdx, first;
    char devName[100];
    int read_config = 0;
    int utc_mode, rcv_mode, dyn_mode;
    int cmd_flags = 0;
    type = 0;

    sprintf(devName, "%s", DEVICE);

    utc_mode = rcv_mode = dyn_mode = -1;
    while ((optc = getopt(argc, argv, "hs:d:pr:u:f")) != -1)
        switch (optc) {
        case 'h':
            usage(argv[0]);
            return 0;
        case 's':
            sprintf(devName, "%s", optarg);
            break;
        case 'p':
            read_config = 1;
            break;
        case 'r':
            rcv_mode = atoi(optarg);
            cmd_flags |= CMD_RECEIVE;
            break;
        case 'u':
            utc_mode = atoi(optarg);
            cmd_flags |= CMD_UTC;
            break;
        case 'd':
            dyn_mode = atoi(optarg);
            cmd_flags |= CMD_DYNAMICS;
            break;
        case 'f':
            cmd_flags |= CMD_FLASH;
            break;
        case '?':
        default:
            usage(argv[0]);
            return -1;
        }

    /* Exit on CTRL-C */
    /* set up signal handling */
    sa_new.sa_handler = quit;
    sigemptyset(&sa_new.sa_mask);
    sa_new.sa_flags = 0;
    sigaction(SIGINT, &sa_new, &sa_old );

    fd = open(devName, O_RDWR | O_NOCTTY ); 
    if (fd < 0) {
        perror(devName); 
        return -1;
    }
    
    tcgetattr(fd,&oldtio); /* save current port settings */
    memset(&newtio, '\0', sizeof(newtio));
    newtio.c_cflag = BAUDRATE | CS8 | CLOCAL | CREAD | PARODD | PARENB;
    newtio.c_iflag = IGNPAR;
    newtio.c_oflag = 0;
    
    /* set input mode (non-canonical, no echo,...) */
    newtio.c_lflag = 0;    
    newtio.c_cc[VTIME]    =  0;
    newtio.c_cc[VMIN]     = 10;
    
    tcflush(fd, TCIFLUSH);
    tcsetattr(fd,TCSANOW,&newtio);

    c = lastc = lastlastc = 0;
    mIdx = first = 0;
    
    /* Read until CTRL-C or done */
    new_mode = mode = STARTUP;
    while (mode != STOP) {
        res = read(fd,buf,255);
        
        /* Loop over characters and parse out messages */
        for (i=0; i<res; i++) {
            c = buf[i];
            /* Look for EOM */
            if ((lastlastc == TSIP_DLE) && (lastc == TSIP_ETX) && (c == TSIP_DLE)) {
                /* Discard first message -- usually incomplete */
                if (!first) {
                    first = 1;
                }            
                else
                    type = tsip_decode(m, mIdx);
                mIdx = 0;                    
            }
            else {
                if (mIdx == MAX_MSG) {
                    fprintf(stderr, "ERROR: maximum message size reached (%d)\n", 
                            MAX_MSG);
                    return -1;
                }
            }

            m[mIdx++] = c;
            lastlastc = lastc;
            lastc = c;
        }

        /* Send commands according to mode */
        switch (mode) {
            
        case STARTUP:
            if (first) 
                new_mode = RUN;
            break;
            
        case RUN:
            if (read_config)
                new_mode = GET_PARAMS;
            else if (cmd_flags != 0)
                new_mode = COMMAND;
            break;
            
        case GET_PARAMS:
            tsip_cmd_getconfig(fd);
            new_mode = WAIT_PARAMS;
            break;

        case WAIT_PARAMS:
            if (type == TSIP_REP_CFG) {
                if (read_config)
                    new_mode = STOP;
                else
                    new_mode = RUN;
            }
            break;

        case COMMAND:
            if (cmd_flags & CMD_RECEIVE) {
                tsip_cmd_set_rcv_mode(fd, rcv_mode);
                cmd_flags &= ~CMD_RECEIVE;
            }
            else if (cmd_flags & CMD_UTC) {
                tsip_cmd_set_utc_mode(fd, utc_mode);
                cmd_flags &= ~CMD_UTC;
            }
            else if (cmd_flags & CMD_DYNAMICS) {
                tsip_cmd_set_dynamics_mode(fd, dyn_mode);
                cmd_flags &= ~CMD_DYNAMICS;
            }
            else if (cmd_flags & CMD_FLASH) {
                tsip_cmd_flash(fd);
                cmd_flags &= ~CMD_FLASH;
                sleep(3);
            }
            new_mode = GET_PARAMS;
            break;
        }

        usleep(50000);
        mode = new_mode;
    }

    tcsetattr(fd,TCSANOW,&oldtio);    
    close(fd);
    return 0;

}


