!> \file binding_energy.f90  This file contains the routines to read in the data files and compute the envelope binding energy for
!!                           a given set of stellar parameters.
!! 
!! \mainpage Binding energy documentation
!! This code reads the data files and computes the envelope binding energy for a given set of stellar parameters.
!!
!! Available metallicities are 1e-4, 1e-3, 0.01, 0.015, 0.02 and 0.03.  For other values, the closest available value in log(Z)
!! will be used.
!! 
!! \see 
!! - Loveridge et al. ApJ 743, 49 (2011) - http://adsabs.harvard.edu/abs/2011ApJ...743...49L
!! - http://www.astro.ru.nl/~sluys/index.php?title=be


!***********************************************************************************************************************************
!> \brief Provides the double-precision kind "double" or "dbl".
module kinds
   implicit none
   !> Double-precision kind
   integer, parameter :: double = selected_real_kind(15,307)
   !> Double-precision kind
   integer, parameter :: dbl = double
end module kinds
!***********************************************************************************************************************************


!***********************************************************************************************************************************
!> \brief This module distributes data between the binding-energy routines;  it does not have to be USEd in the calling program.
module BE_data
   use kinds
   
   implicit none
   
   ! Metallicities:
   !> Number of available metallicities
   integer, parameter :: nz = 6
   !> List of available metallicities
   real(double), parameter :: zs(nz) = &
        (/1.e-4_dbl, 1.e-3_dbl, 0.01_dbl, 0.015_dbl, 0.02_dbl, 0.03_dbl/)
   
   ! Model groups:
   !> Number of model groups
   integer, parameter :: ngr=5
   !> List of model groups
   character, parameter :: groups(ngr)*(5) = (/'LMR1 ','LMR2 ','LMA  ','HM   ','Recom'/)
   
   
   ! Number of coefficients per Z, group:
   !> Maximum number of coefficients
   integer, parameter :: ndatmax=999
   !> Actual number of coefficients
   integer :: ndat(nz,ngr)
   
   
   !> Fitting coefficients (alpha_m,r)
   real(double) :: alphas(nz,ngr,ndatmax)
   
   
   !> Powers for M for each coefficient
   integer :: ms(nz,ngr,ndatmax)
   !> Powers for R for each coefficient
   integer :: rs(nz,ngr,ndatmax)
   
   
   !> Low-mass/high-mass boundaries (boundaries between groups LM* and HM)
   real(double) :: LMHMbs(nz)
   
   ! Coefficients for RGB boundaries (boundaries between groups LMR1 and LMR2):
   !> Number of coefficients for RGB boundaries (boundaries between groups LMR1 and LMR2)
   integer, parameter :: nRGBbc = 5
   !> Coefficients for RGB boundaries (boundaries between groups LMR1 and LMR2)
   real(double) :: RGBb_coef(nz,0:nRGBbc-1)
   
   
   
   !> The binding energy was originally expressed in erg per solar mass to avoid large numbers, 
   !! so we need to add this constant (log10[Mo/g]) to log(BE).
   real(double), parameter :: logBE0 = 33.29866_dbl                                   ! log10(1.9891e33) = 33.29866
   
   
end module BE_data
!***********************************************************************************************************************************



!***********************************************************************************************************************************
!> \brief This module contains the subroutines and functions that are called to compute the binding energy for a set of stellar
!!        parameters.  This module should be USEd by the calling program.
module calc_BE
  implicit none
  private
  save
  
  public :: read_BE_input, read_BE_input_Z, calc_logBE, calc_logBE_recom
  
contains
  
  !*********************************************************************************************************************************
  !> \brief This subroutine reads the input files for all available metallicities.
  !!        Call this routine once when starting your program, before calling the function calc_logBE() or calc_logBE_recom().
  !!
  !! \param path   The path (relative or absolute) where the input data files can be found.
  !! \retval None  The data that are read in are distributed via the module BE_data.
  
  subroutine read_BE_input(path)
    use kinds
    use BE_data
    
    implicit none
    character, intent(in) :: path*(*)
    integer :: iz
    
    ms = 0
    rs = 0
    alphas = 0.0_dbl
    
    do iz=1,nz
       call read_BE_input_Z(path,iz)
    end do
    
  end subroutine read_BE_input
  !*********************************************************************************************************************************
  
  
  
  !*********************************************************************************************************************************
  !>  \brief This subroutine reads the input file for the iz-th metallicity in the database.
  !!  It is called by read_BE_input(), and in principle there is no need for the user to call it directly.
  !!
  !!  \param path   The path (relative or absolute) where the input data files can be found;
  !!  \param iz     Number of the desired metallicity in the database.
  !!  \retval None  The data that are read in are distributed via the module BE_data.
  
  subroutine read_BE_input_Z(path,iz)
    use kinds
    use BE_data
    
    implicit none
    character, intent(in) :: path*(*)
    integer, intent(in) :: iz
    
    integer :: ip,ig,i,io, verbose
    real(double) :: z,z0
    character :: infile*(99),tmpstr*(99),group*(9)
    
    
    ip = 10      ! Input unit
    
    z = zs(iz)   ! Value of the current metallicity, the iz-th Z in the database (Z=0.02 = solar)
    
    verbose = 1  ! Verbosity of the output: 0-none (not even errors!), 1-error messages (default), 2-3-additional info
    
    
    ! Compose the input file name and open the file:
    write(infile,'(A,I6.6,A4)')trim(path)//'/z', nint(z*1.e5_dbl),'.dat'
    if(verbose.ge.2) write(6,'(A)')'  Reading data file '//trim(infile)//'...'
    open(unit=ip,form='formatted',status='old',action='read',position='rewind',file=trim(infile),iostat=io)
    if(io.ne.0) then
       if(verbose.ge.1) write(0,'(A,/)')'  Error opening '//trim(infile)//', aborting...'
       stop
    end if
    
    
    ! Read the empty line:
    read(ip,'(A)')tmpstr
    
    
    ! Read the LM/HM cut mass:
    read(ip,'(F10.5)',iostat=io)z0
    if(io.ne.0) then
       if(verbose.ge.1) write(0,'(A)')'  Error reading file '//trim(infile)//', LM/HM cut, aborting...'
       stop
    end if
    if(abs(z0-z)/z.gt.1.e-6_dbl) then
       if(verbose.ge.1) then
          write(0,'(A)')'  Error:  reading the wrong metallicity from the input file '//trim(infile)//', LM/HM cut, aborting...'
          write(0,'(2(A,F7.5),A,/)')'          I need Z=',z,', but find Z=',z0,'.'
       end if
       stop
    end if
    read(ip,*)LMHMbs(iz)
    if(LMHMbs(iz).gt.0.0_dbl) then
       LMHMbs(iz) = log10(LMHMbs(iz))
    else
       if(verbose.ge.1) write(0,'(A)')'  Error reading LMHMbs, aborting...'
       stop
    end if
    
    
    ! Read the empty line:
    read(ip,'(A)')tmpstr
    
    
    ! Read the LMR1/LMR2 cut coefficients:
    read(ip,'(F10.5)',iostat=io)z0
    if(io.ne.0) then
       if(verbose.ge.1) write(0,'(A)')'  Error reading file '//trim(infile)//', LMR1/LMR2 cut, aborting...'
       stop
    end if
    if(abs(z0-z)/z.gt.1.e-6_dbl) then
       if(verbose.ge.1) then
          write(0,'(A)')'  Error:  reading the wrong metallicity from the input file '//trim(infile)// &
               ', LMR1/LMR2 cut, aborting...'
          write(0,'(2(A,F7.5),A,/)')'          I need Z=',z,', but find Z=',z0,'.'
       end if
       stop
    end if
    read(ip,*)rgbb_coef(iz,0:nrgbbc-1)
    
    
    ! Read the empty line:
    read(ip,'(A)')tmpstr
    
    
    ! Read the rest of the input file group by group (1-4: LMR1, LMR1, LMA, HM):
    do ig=1,ngr
       
       ! Read the metallicity and group name:
       if(verbose.ge.3) write(6,'(A)', advance='no')'    reading group '//trim(groups(ig))//'...'
       read(ip,'(F10.5,5x,A)',iostat=io)z0,group
       if(io.ne.0) then
          if(verbose.ge.1) write(0,'(A)')'  Error reading file '//trim(infile)//', group '//trim(groups(ig))//', aborting...'
          stop
       end if
       if(abs(z0-z)/z.gt.1.e-6_dbl) then
          if(verbose.ge.1) then
             write(0,'(A)')'  Error:  reading the wrong metallicity from the input file '//trim(infile)// &
               ' in group '//trim(groups(ig))//', aborting...'
             write(0,'(2(A,F7.5),A,/)')'          I need Z=',z,', but find Z=',z0,'.'
          end if
          stop
       end if
       if(trim(group).ne.trim(groups(ig))) then
          if(verbose.ge.1) then
             write(0,'(A)')'  Error:  reading the wrong group from the input file '//trim(infile)//', aborting...'
             write(0,'(A,/)')'          I need '//trim(groups(ig))//', but find '//trim(group)//'.'
          end if
          stop
       end if
       
       
       ! Read the header line:
       read(ip,'(A)')tmpstr  ! 'm','r','alpha_mr'
       
       
       ! Read the coefficients:
       do i=1,ndatmax
          if(i.ge.ndatmax) then
             if(verbose.ge.1) write(0,'(A)')'  Error:  arrays too small.  Please increase ndatmax.  Aborting...'
             stop
          end if
          read(ip,'(2I6,ES30.20)',iostat=io) ms(iz,ig,i), rs(iz,ig,i), alphas(iz,ig,i)
          
          if(io.gt.0) exit
          if(ms(iz,ig,i).eq.0 .and. rs(iz,ig,i).eq.0 .and. abs(alphas(iz,ig,i)).lt.tiny(0.0_dbl)) exit
          if(io.lt.0) then
             if(verbose.ge.1) write(0,'(A)')'  Error:  EOF reached unexpectedly while reading the input file '//trim(infile)// &
                  ', aborting...'
             stop
          end if
          
          ndat(iz,ig) = i
       end do  ! i
       
       if(verbose.ge.3) write(6,'(I6,A)')ndat(iz,ig),' lines read.'
    end do  ! ig - group
    
    
    ! Close the input file:
    close(ip)
    
  end subroutine read_BE_input_Z
  !*********************************************************************************************************************************
  
  
  
  !*********************************************************************************************************************************
  !> \brief This function computes log[BE/erg] as a function of log[Z], Mzams, M, log[R/Ro] and GB.
  !!
  !! \param logZ   10-base log of metallicity (solar = log[0.02]);
  !! \param Mzams  stellar ZAMS mass (in solar masses);
  !! \param M      current stellar mass (in solar masses);
  !! \param logR   10-base log of stellar radius, expressed in solar radii (log[R/Ro]);
  !! \param GB     giant branch: 1: RGB, 2: AGB (CO core exists).
  !! \param ignore_massloss  if true, ignore the factor Lambda that corrects for mass loss (logical, optional; default: false)
  !! \retval calc_logBE  The 10-base log of the absolute value of the envelope binding energy expressed in erg.
  
  function calc_logBE(logZ, Mzams, M, logR, GB,  ignore_massloss)
    use kinds
    use BE_data
    
    implicit none
    integer, intent(in) :: GB
    real(double), intent(in) :: logZ,Mzams,M,logR
    logical, optional, intent(in) :: ignore_massloss
    integer :: ii,iz,ig
    real(double) :: calc_logBE,logM,dz,dzi,logRbound,dlogBE,lambda
    
    logM = log10(M)
    iz = 0
    
    ! Find the nearest logZ in the grid:
    dz = huge(dz)
    do ii=1,nz
       dzi = abs(log10(zs(ii))-logZ)
       if(dzi.lt.dz) then
          dz = dzi
          iz = ii
       end if
    end do
    
    
    
    ! Find the group from the mass and evolutionary state (ig = 1-4):
    if(logM.le.LMHMbs(iz)) then                           ! group LM (low mass)
       if(GB.eq.1) then                                   ! group LMR (low-mass RGB)
          ig = 1                                          ! group LMR1 (low-mass RGB 1)
          
          ! Compute boundary radius between LMR1 an LMR2:
          logRbound = 0.0_dbl
          do ii=0,nRGBbc-1
             logRbound = logRbound + RGBb_coef(iz,ii)*logM**ii
          end do
          if(logR.gt.logRbound) ig = 2                    ! group LMR2 (low-mass RGB 2)
       else                                               ! group LMA (low-mass AGB)
          ig = 3
       end if
    else                                                  ! group HM (high mass)
       ig = 4
    end if
    
    
    ! Compute the 10-logarithm of the binding energy:
    calc_logBE = 0.0_dbl
    do ii=1,ndat(iz,ig)
       dlogBE = alphas(iz,ig,ii) * (logM+tiny(logM))**dble(ms(iz,ig,ii)) * (logR+tiny(logR))**dble(rs(iz,ig,ii))  ! Avoid 0**0
       calc_logBE = calc_logBE + dlogBE
    end do
    
    
    ! Compute and apply the mass-loss correction factor Lambda:
    lambda = 1.0_dbl + 0.25_dbl * ((Mzams-M)/Mzams)**2
    if(present(ignore_massloss)) then
       if(ignore_massloss) lambda = 1.0_dbl  ! If the optional parameter ignore_massloss is present and true, set Lambda=1
    end if
    
    calc_logBE = lambda * calc_logBE
    
    
    ! BE was originally expressed in erg/solar mass to avoid large numbers, so we need to convert to erg here:
    calc_logBE = calc_logBE + logBE0
    
  end function calc_logBE
  !*********************************************************************************************************************************
  
  
  !*********************************************************************************************************************************
  !> \brief This function computes the recombination-energy term log[BE_recom/erg] as a function of log[Z], M, and log[R/Ro].
  !!
  !! \param logZ   10-base log of metallicity (solar = log[0.02]);
  !! \param M      current stellar mass (in solar masses);
  !! \param logR   10-base log of stellar radius, expressed in solar radii (log[R/Ro]);
  !! \retval calc_logBE  The 10-base log of the absolute value of the envelope binding energy expressed in erg.
  
  function calc_logBE_recom(logZ, M, logR)
    use kinds
    use BE_data
    
    implicit none
    real(double), intent(in) :: logZ,M,logR
    integer :: ii,iz,ig
    real(double) :: calc_logBE_recom, logM,dz,dzi,dlogBE
    
    logM = log10(M)
    iz = 0
    
    ! Find the nearest logZ in the grid:
    dz = huge(dz)
    do ii=1,nz
       dzi = abs(log10(zs(ii))-logZ)
       if(dzi.lt.dz) then
          dz = dzi
          iz = ii
       end if
    end do
    
    
    
    ! The recombination-energy term has ID 5:
    ig = 5
    
    
    ! Compute the 10-logarithm of the binding energy:
    calc_logBE_recom = 0.0_dbl
    do ii=1,ndat(iz,ig)
       dlogBE = alphas(iz,ig,ii) * (logM+tiny(logM))**dble(ms(iz,ig,ii)) * (logR+tiny(logR))**dble(rs(iz,ig,ii))  ! Avoid 0**0
       calc_logBE_recom = calc_logBE_recom + dlogBE
    end do
    
    
    ! BE was originally expressed in erg/solar mass to avoid large numbers, so we need to convert to erg here:
    calc_logBE_recom = calc_logBE_recom + logBE0
    
  end function calc_logBE_recom
  !*********************************************************************************************************************************
  
  
end module calc_BE
!***********************************************************************************************************************************

